# Add Custom Model

Add a custom model to the Voyager SDK model zoo for Axelera Metis AI hardware

## Important Context

**This command adds models specifically for deployment on Axelera Metis AI Processing Units (APUs).**
All models are configured for the Voyager SDK compilation and inference toolchain.

## Instructions

Add the specified model: **$ARGUMENTS**

### Step 0: Data Source & Environment Selection

**FIRST**, use the `AskUserQuestion` tool to determine the setup:

```
Question: "Which Voyager SDK setup would you like to use for adding the model?"
Options:
1. "ChipOS Knowledge Base Only" - Use RAG for research and model integration examples only (no local execution)
2. "ChipOS Knowledge Base + Local SDK" - Use RAG for model integration examples, add to local SDK
3. "Local Voyager SDK Repository" - Use an existing local installation only
4. "Clone Fresh Repository" - Clone latest Voyager SDK for model development
```

**Based on selection:**

**If ChipOS Knowledge Base Only:**
- Search for model integration information and documentation:
  ```python
  mcp__chipos__rag_search_code_examples(
      query="model YAML configuration ONNX PyTorch",
      source_id="github.com/axelera-ai-hub/voyager-sdk",
      match_count=5
  )
  mcp__chipos__rag_search_knowledge_base(
      query="add custom model zoo integration",
      source_id="github.com/axelera-ai-hub/voyager-sdk",
      match_count=5
  )
  ```
- Provide research-based guidance and documentation
- No local SDK execution required - purely informational

**If ChipOS Knowledge Base + Local SDK:**
- Search for model integration examples:
  ```python
  mcp__chipos__rag_search_code_examples(
      query="model YAML configuration ONNX PyTorch",
      source_id="github.com/axelera-ai-hub/voyager-sdk",
      match_count=5
  )
  mcp__chipos__rag_search_knowledge_base(
      query="add custom model zoo",
      source_id="github.com/axelera-ai-hub/voyager-sdk",
      match_count=5
  )
  ```
- Ask for local SDK path to add the model
- Set environment variables accordingly

**If Local Repository:**
- Ask for the repository path: "What is the path to your Voyager SDK installation?"
- Set environment:
  ```bash
  export AXELERA_FRAMEWORK=/path/to/voyager-sdk
  export AXELERA_RUNTIME_DIR=$AXELERA_FRAMEWORK/runtime
  source $AXELERA_FRAMEWORK/venv/bin/activate
  ```
- Review existing models in `$AXELERA_FRAMEWORK/ax_models/zoo/`

**If Clone Fresh:**
- Clone and install:
  ```bash
  git clone https://github.com/axelera-ai-hub/voyager-sdk.git
  cd voyager-sdk
  ./install.sh --all --media
  source venv/bin/activate
  export AXELERA_FRAMEWORK=$(pwd)
  ```

### Step 0.5: ChipOS Project & Task Integration

**Check if project context is already set (from previous commands or `/init-chipos`).**

If NO project context exists, ask:
```
Question: "Would you like to track this model integration in ChipOS?"
Options:
1. "Yes, use existing project" - Select from available ChipOS projects
2. "Yes, create new project" - Create a new project for this work
3. "No, skip task tracking" - Proceed without ChipOS task management
```

**If using existing project:**
```python
mcp__chipos__list_projects()
# User selects project
```

**If creating new project:**
```python
mcp__chipos__create_project(
    title="Voyager SDK Model - [Model Name]",
    description="Add custom model: $ARGUMENTS",
    github_repo="https://github.com/axelera-ai-hub/voyager-sdk"
)
```

**Create model integration task:**
```python
mcp__chipos__create_task(
    project_id="[PROJECT_ID]",
    title="Add model: [Model name from $ARGUMENTS]",
    description="Integrate custom model into model zoo: $ARGUMENTS",
    assignee="AI IDE Agent",
    feature="model-integration",
    task_order=10
)
mcp__chipos__update_task(task_id="[TASK_ID]", status="doing")
```

### Step 1: Model Source Identification
   Determine model source format:
   - **PyTorch**: `.pt`, `.pth` files or torchvision/timm model name
   - **ONNX**: `.onnx` file
   - **TensorFlow**: Convert to ONNX first
   - **Custom**: Define custom model class

2. **Model Requirements Analysis**
   Gather model information:
   - Input shape (batch, channels, height, width)
   - Input format (RGB/BGR, NCHW/NHWC)
   - Normalization parameters (mean, std)
   - Output format and shape
   - Model task (detection, classification, etc.)

3. **ONNX Model Integration**
   For ONNX models:
   ```yaml
   # ax_models/zoo/custom/<model-name>-onnx.yaml

   network:
     name: <model-name>-onnx
     description: <description>

   model:
     source: onnx
     path: /path/to/model.onnx
     # Or URL:
     url: https://example.com/model.onnx

   input:
     shape: [1, 3, 640, 640]
     layout: NCHW
     dtype: float32

   preprocessing:
     - type: AxResize
       size: [640, 640]
       mode: letterbox
     - type: AxNormalize
       mean: [0.0, 0.0, 0.0]
       std: [255.0, 255.0, 255.0]

   postprocessing:
     # Add appropriate decoder

   calibration:
     dataset: imagenet  # or custom
     num_samples: 100
   ```

4. **PyTorch Model Integration**
   For PyTorch models:
   ```yaml
   # ax_models/zoo/torch/<model-name>.yaml

   network:
     name: <model-name>

   model:
     source: torch
     module: ax_models.torch.<module_name>
     class: <ClassName>
     weights: /path/to/weights.pt
     # Or download URL:
     weights_url: https://example.com/weights.pt

   input:
     shape: [1, 3, 224, 224]
     layout: NCHW
     dtype: float32

   preprocessing:
     - type: AxResize
       size: [224, 224]
     - type: AxNormalize
       mean: [0.485, 0.456, 0.406]
       std: [0.229, 0.224, 0.225]

   postprocessing:
     - type: Softmax
     - type: TopK
       k: 5
   ```

5. **Timm Model Integration**
   For timm (PyTorch Image Models):
   ```yaml
   # ax_models/zoo/timm/<model-name>-imagenet.yaml

   network:
     name: <model-name>-imagenet

   model:
     source: timm
     model_name: <timm_model_name>
     pretrained: true
     num_classes: 1000

   input:
     shape: [1, 3, 224, 224]
     layout: NCHW
     dtype: float32

   preprocessing:
     - type: AxResize
       size: [224, 224]
     - type: AxNormalize
       mean: [0.485, 0.456, 0.406]
       std: [0.229, 0.224, 0.225]

   postprocessing:
     - type: Classification
       labels: imagenet1k
   ```

6. **Ultralytics YOLO Integration**
   For YOLO models from Ultralytics:
   ```yaml
   # ax_models/zoo/yolo/<model>-coco.yaml

   network:
     name: <model>-coco

   model:
     source: ultralytics
     model_name: <yolov8n|yolov8s|yolov8m|...>
     weights: /path/to/custom.pt  # Optional custom weights

   input:
     shape: [1, 3, 640, 640]
     layout: NCHW
     dtype: float32

   preprocessing:
     - type: AxResize
       size: [640, 640]
       mode: letterbox

   postprocessing:
     - type: YoloDecoder
       version: 8
       num_classes: 80
       conf_threshold: 0.25
       iou_threshold: 0.45

   calibration:
     dataset: coco2017
     num_samples: 100
   ```

7. **Custom Model Class**
   For models requiring custom code:
   ```python
   # ax_models/torch/<model_name>.py

   import torch
   import torch.nn as nn

   class CustomModel(nn.Module):
       def __init__(self, num_classes=1000):
           super().__init__()
           # Define model architecture
           self.features = nn.Sequential(...)
           self.classifier = nn.Linear(512, num_classes)

       def forward(self, x):
           x = self.features(x)
           x = self.classifier(x)
           return x

       @classmethod
       def from_pretrained(cls, weights_path):
           model = cls()
           state_dict = torch.load(weights_path, map_location='cpu')
           model.load_state_dict(state_dict)
           return model
   ```

8. **Custom Decoder**
   If model requires custom postprocessing:
   ```python
   # ax_models/decoders/<task>.py

   from axelera.app.meta import Detection, BBox

   class CustomDecoder:
       def __init__(self, conf_threshold=0.5, num_classes=80):
           self.conf_threshold = conf_threshold
           self.num_classes = num_classes

       def decode(self, outputs, image_shape):
           """
           Args:
               outputs: Model output tensors
               image_shape: Original image shape (H, W)
           Returns:
               List of Detection objects
           """
           detections = []
           # Parse model outputs
           # Create Detection objects
           return detections
   ```

9. **Custom Dataset/Calibration**
   For models with custom calibration data:
   ```python
   # ax_datasets/<dataset_name>.py

   from axelera.app.data_utils import CalibrationDataset

   class CustomCalibrationDataset(CalibrationDataset):
       def __init__(self, data_path, num_samples=100):
           self.data_path = data_path
           self.num_samples = num_samples
           self.samples = self._load_samples()

       def _load_samples(self):
           # Load and preprocess calibration images
           pass

       def __len__(self):
           return min(len(self.samples), self.num_samples)

       def __getitem__(self, idx):
           return self.samples[idx]
   ```

10. **Model Validation**
    Before adding to zoo, validate:
    ```bash
    # Test deployment
    ./deploy.py <model-name> --mode quantize

    # Test inference
    ./inference.py <model-name> <test-image> --no-display

    # Check accuracy
    ./inference.py <model-name> dataset --no-display
    ```

11. **File Locations**
    - Model YAML: `ax_models/zoo/<category>/<model-name>.yaml`
    - Custom model code: `ax_models/torch/<module>.py`
    - Custom decoder: `ax_models/decoders/<task>.py`
    - Custom dataset: `ax_datasets/<dataset>.py`
    - Custom evaluator: `ax_evaluators/<task>.py`

12. **Documentation**
    After adding model:
    - Add entry to model zoo documentation
    - Include expected accuracy metrics
    - Document any special requirements
    - Add example usage in comments
