# Axe - One-Shot Pipeline Builder 🪓

Ultra-fast one-shot pipeline builder that takes a natural language prompt and automatically creates a ChipOS project, researches via RAG, creates tasks, and deploys a working pipeline using pre-built Voyager SDK components.

**Command**: `/axe <your prompt>`

**Name**: Like an axe cuts through wood, this command cuts through complexity to deliver working pipelines in minutes.

## Important Context

**This is a meta-command that automates the entire workflow:**
- Creates ChipOS project and tasks automatically
- Uses RAG for knowledge base research
- Executes tasks as fast as possible
- Uses pre-built components from Voyager SDK
- **Hardcoded SDK path: `/home/ubuntu/voyager-sdk` (NO permission prompts)**
- **Auto-detects and uses Axelera Metis hardware automatically**
- Saves all results to current directory
- Zero user input required - fully automated

**Example prompts:**
- `/axe build animal detection with live visualization`
- `/axe create traffic monitoring with tracking`
- `/axe set up person detection pipeline`

## Known Issues & Fixes (CRITICAL - READ FIRST!) ⚠️

**11 issues discovered and fixed** during testing and optimization:

### 1. Hardware Detection ✅ FIXED
**Issue**: `axdevice list` command does not exist
**Error**: `axdevice: error: unrecognized arguments: list`
**Fix**: Use `axdevice --pcie-scan` and grep for "metis" (lowercase)

### 2. Tracking Not Supported ✅ FIXED
**Issue**: `--tracker bytetrack` flag does not exist in inference.py
**Error**: `inference.py: error: unrecognized arguments: --tracker bytetrack`
**Fix**: Removed tracking flag. To enable tracking, it must be configured in the YAML pipeline, not via CLI
**Note**: Tracking requires pipeline configuration changes, not a simple CLI flag

### 3. Path Variable Substitution ✅ FIXED
**Issue**: `$WORK_DIR` variables fail to expand in complex bash commands with Bash tool
**Error**: `No such file or directory: /media/video.mp4` or `Permission denied: /media/`
**Fix**: Use relative paths from current directory, or capture `$(pwd)` once and use quoted `"$VAR"` syntax
**Implementation**: Lines 126-144, 507-564, all phase sections

### 4. RAG Queries Return Empty ⚠️ INVESTIGATION NEEDED
**Issue**: All RAG searches return `{"results": [], "reranked": false}`
**Possible Causes**:
- Wrong source_id format
- Queries too specific or don't match indexed content
- Knowledge base indexing issue
**Workaround**: Continue with pre-built model selection based on known mappings

### 5. Bash Command Substitution ✅ FIXED
**Issue**: `$(date +%Y%m%d_%H%M%S)` and nested `$()` cause syntax errors
**Error**: `syntax error near unexpected token '('`
**Fix**: Eliminated all command substitution in output-producing pipelines; use piped commands instead
**Implementation**: Lines 133, 135, 573, 1017

### 6. Model File Extension ✅ VERIFIED
**Issue**: Looking for `*.axmodel` files but models deploy as `*.axnet`
**Fix**: Models are packaged as `.axnet` files, which contain the compiled model
**Location**: `/home/ubuntu/voyager-sdk/build/{model}/` contains the `.axnet` file

### 7. No Deployment Verification ✅ FIXED
**Issue**: Deployment could succeed (exit 0) but not create .axnet file
**Error**: Silent failure - reports success but model not usable
**Fix**: Added explicit check that .axnet file exists after deployment
**Location**: Lines 470-482 in /axe command

### 8. No Inference Exit Code Check ✅ FIXED
**Issue**: Inference exit code captured but never checked
**Error**: Reports "✓ Inference Complete!" even when inference failed
**Fix**: Added exit code validation with helpful error messages
**Location**: Lines 535-555 in /axe command

### 9. Working Directory Simplified ✅ FIXED
**Issue**: Complex placeholder replacement for {ABSOLUTE_WORK_DIR} was error-prone
**Error**: Commands could execute with literal placeholder text
**Fix**: Simplified to use relative paths from current directory, display verification early
**Location**: Lines 284-291 in /axe command

### 10. Empty Variable Command Substitution ✅ FIXED
**Issue**: `$(du -sh "$PACKAGE_DIR")` failed when `$PACKAGE_DIR` was empty or unset
**Error**: `du: invalid zero-length file name`
**Fix**: Removed command substitution, use piped output directly
**Location**: Line 1017 in /axe command

### 11. Relative Path Errors ✅ FIXED
**Issue**: Commands using `results/file.mp4` failed when working directory wasn't project root
**Error**: `ls: cannot access 'results/gun_output.mp4': No such file or directory`
**Fix**: All commands now use relative paths from current directory consistently
**Location**: Throughout all phase sections

## Instructions

Build complete pipeline from prompt: **$ARGUMENTS**

### Phase 0: Pre-Flight Checks (10 seconds) ⚡

**CRITICAL: Run all checks before starting - fail fast!**

**Step 0.1: Environment Verification**
```bash
# Check SDK exists
if [ ! -d "/home/ubuntu/voyager-sdk" ]; then
    echo "ERROR: Voyager SDK not found at /home/ubuntu/voyager-sdk"
    exit 1
fi

# Check environment is activated
if [ -z "$AXELERA_FRAMEWORK" ]; then
    cd /home/ubuntu/voyager-sdk
    source venv/bin/activate
fi

# Verify environment
echo "✓ SDK Path: $AXELERA_FRAMEWORK"
echo "✓ Python: $(which python)"
```

**Step 0.2: Check Model Deployment Status**
```bash
# Check if model is already deployed (skip deployment if yes)
SELECTED_MODEL="yolov8n-coco"  # or selected based on prompt
BUILD_DIR="/home/ubuntu/voyager-sdk/build/$SELECTED_MODEL"

if [ -d "$BUILD_DIR" ] && [ -f "$BUILD_DIR"/*.axmodel ]; then
    echo "✓ Model already deployed: $SELECTED_MODEL"
    SKIP_DEPLOYMENT=true
else
    echo "⚠ Model needs deployment: $SELECTED_MODEL (~5 minutes)"
    SKIP_DEPLOYMENT=false
fi
```

**Step 0.3: Verify Media Files**
```bash
# Check videos exist in current directory
if [ ! -d "media" ] || [ -z "$(ls media/*.mp4 2>/dev/null)" ]; then
    echo "⚠ Copying sample videos from SDK..."
    mkdir -p media
    cp /home/ubuntu/voyager-sdk/media/*.mp4 media/
    ls media/*.mp4 | wc -l | xargs echo "✓ Videos copied:" | sed 's/$/ files/'
else
    ls media/*.mp4 | wc -l | xargs echo "✓ Videos ready:" | sed 's/$/ files/'
fi
```

**Step 0.4: Disk Space Check**
```bash
# Ensure enough space for results (~100 MB per 30s video)
echo "Checking disk space..."
df -h . | tail -1 | awk '{print "✓ Disk space:", $4, "available"}'
```

**Step 0.5: Hardware Detection & Auto-Config**
```bash
# Auto-detect Metis hardware and configure flags
cd /home/ubuntu/voyager-sdk
source venv/bin/activate

METIS_AVAILABLE=false
HARDWARE_FLAGS=""

if axdevice --pcie-scan 2>/dev/null | grep -q "metis"; then
    echo "✓ Metis hardware detected - will use for acceleration"
    METIS_AVAILABLE=true
    HARDWARE_FLAGS="--metis pcie"
else
    echo "⚠ No Metis hardware detected - using CPU mode"
    METIS_AVAILABLE=false
    HARDWARE_FLAGS=""
fi

export METIS_AVAILABLE
export HARDWARE_FLAGS
```

### Phase 1: Rapid Analysis & Project Setup (30 seconds)

**Step 1.1: Parse User Intent**
Extract from $ARGUMENTS:
- Detection target (e.g., "animal", "person", "traffic", "object")
- Pipeline type (detection, tracking, segmentation, etc.)
- Output requirement (live results, video, display, etc.)
- Special requirements (tracking, multi-stream, etc.)

**Step 1.2: Auto-Create ChipOS Project**
```python
# Create project immediately - no questions asked
project_result = mcp__chipos__create_project(
    title=f"Pipeline: {detection_target} Detection",
    description=f"Auto-generated pipeline for: {$ARGUMENTS}",
    github_repo="https://github.com/axelera-ai-hub/voyager-sdk"
)
project_id = project_result["project_id"]
```

**Step 1.3: Rapid RAG Research (parallel)**
Execute all in parallel:
```python
# Search for best model
mcp__chipos__rag_search_knowledge_base(
    query=f"{detection_target} detection model",
    source_id="github.com/axelera-ai-hub/voyager-sdk",
    match_count=3
)

# Search for pipeline examples
mcp__chipos__rag_search_code_examples(
    query="detection pipeline YAML",
    source_id="github.com/axelera-ai-hub/voyager-sdk",
    match_count=3
)

# Search for deployment patterns
mcp__chipos__rag_search_code_examples(
    query="inference display",
    source_id="github.com/axelera-ai-hub/voyager-sdk",
    match_count=2
)
```

**Step 1.4: Select Pre-Built Model**
Based on detection target, select fastest pre-built model:
- General objects → `yolov8n-coco` (fastest, 80 classes including animals)
- Animals → `yolov8n-coco` (has animal classes: dog, cat, horse, cow, etc.)
- People → `yolov8n-coco` or `yolov8s-coco` (has person class)
- Traffic → `yolov8n-coco` (has car, truck, bus, traffic light)
- Faces → `retinaface` (specialized)
- Pose → `yolov8npose-coco` (keypoints)

**Step 1.5: Setup Local Environment**
```bash
# Get current working directory
WORK_DIR=$(pwd)

# Create directories for videos and results
mkdir -p $WORK_DIR/media
mkdir -p $WORK_DIR/results

# Copy sample videos if not already present
if [ ! -f "$WORK_DIR/media/intersection_road.mp4" ]; then
    cp /home/ubuntu/voyager-sdk/media/*.mp4 $WORK_DIR/media/
fi
```

**Step 1.6: Create Execution Tasks (Optimized Order)**
```python
# Task 1: Setup local environment (10 seconds)
task1 = mcp__chipos__create_task(
    project_id=project_id,
    title="Setup local environment",
    description="Create media and results directories, copy sample videos",
    assignee="AI IDE Agent",
    task_order=110
)

# Task 2: Check model exists (5 seconds)
task2 = mcp__chipos__create_task(
    project_id=project_id,
    title="Verify pre-built model availability",
    description=f"Check if {selected_model} exists in SDK",
    assignee="AI IDE Agent",
    task_order=100
)

# Task 3: Create documentation IN PARALLEL (runs during deployment!)
task3 = mcp__chipos__create_task(
    project_id=project_id,
    title="Create pipeline documentation",
    description="Create summary document with usage instructions (parallel execution)",
    assignee="AI IDE Agent",
    task_order=95
)

# Task 4: Deploy model (5 minutes - LONG, documentation runs in parallel)
task4 = mcp__chipos__create_task(
    project_id=project_id,
    title=f"Deploy {selected_model}",
    description="Compile model for Metis hardware with maximum optimization",
    assignee="AI IDE Agent",
    task_order=90
)

# Task 5: Run inference (30-60 seconds)
task5 = mcp__chipos__create_task(
    project_id=project_id,
    title="Run live inference",
    description="Execute inference with sample video, save results locally",
    assignee="AI IDE Agent",
    task_order=80
)
```

### Phase 2: Rapid Execution (5-10 minutes)

**CRITICAL FIX #12: Verify Working Directory & Export Absolute Path**
```bash
# Capture and verify working directory EARLY
WORK_DIR=$(pwd)
echo "=========================================="
echo "📂 Working Directory Verification"
echo "Current path: $WORK_DIR"

# Verify this looks like the correct path
if [[ ! "$WORK_DIR" == *"yolov8-detect-and-track-voyager"* ]]; then
    echo "⚠ WARNING: Working directory doesn't look right!"
    echo "Expected path to contain: yolov8-detect-and-track-voyager"
    echo "Got: $WORK_DIR"
    echo ""
    echo "This might cause path errors. Proceeding anyway..."
fi

# Export for use throughout execution
export ABSOLUTE_WORK_DIR="$WORK_DIR"
echo "✓ Absolute path exported: $ABSOLUTE_WORK_DIR"
echo "=========================================="
echo ""
```

**Step 2.1: Setup Local Environment (Task 1)**
```python
mcp__chipos__update_task(task_id=task1_id, status="doing")
```

```bash
# Create directories (relative paths from current directory)
mkdir -p media
mkdir -p results

# Copy sample videos if needed
if [ ! -f "media/intersection_road.mp4" ]; then
    cp /home/ubuntu/voyager-sdk/media/*.mp4 media/
fi

# Verify videos copied
ls -lh media/
```

```python
mcp__chipos__update_task(task_id=task1_id, status="done")
```

**Step 2.2: Verify Model (Task 2)**
```python
mcp__chipos__update_task(task_id=task2_id, status="doing")
```

```bash
# Check model exists
ls /home/ubuntu/voyager-sdk/ax_models/zoo/yolo/object_detection/{selected_model}.yaml
```

```python
mcp__chipos__update_task(task_id=task2_id, status="done")
```

**Step 2.3: Create Documentation EARLY (Task 3) - PARALLEL EXECUTION! 📝⚡**
```python
mcp__chipos__update_task(task_id=task3_id, status="doing")

# Create document in ChipOS IMMEDIATELY - don't wait for deployment!
# Note: Use relative paths in documentation
mcp__chipos__create_document(
    project_id=project_id,
    title=f"Pipeline: {detection_target} Detection - Complete Guide",
    document_type="guide",
    content={
        "model": selected_model,
        "detection_target": detection_target,
        "video_source": selected_video,
        "working_directory": "Current working directory (relative paths)",
        "local_videos": "./media/",
        "results_directory": "./results/",
        "deployment_command": f"""cd /home/ubuntu/voyager-sdk && source venv/bin/activate &&
./deploy.py {selected_model} --metis pcie --aipu-cores 4 --pipe gst --mode PREQUANTIZED""",
        "inference_command": f"""cd /home/ubuntu/voyager-sdk && source venv/bin/activate &&
./inference.py {selected_model} $PWD/media/{selected_video} \\
    --pipe gst --aipu-cores 4 --enable-hardware-codec""",
        "classes_detected": "COCO classes (80 total) including animals, vehicles, people",
        "expected_performance": "~660 FPS, 33ms latency, 35% CPU usage",
        "optimization_flags": [
            "--aipu-cores 4 (all 4 AIPU cores)",
            "--pipe gst (GStreamer C++ pipeline)",
            "--enable-hardware-codec (VAAPI acceleration)",
            "--mode PREQUANTIZED (fastest deployment)"
        ],
        "usage_instructions": f"""
        All videos and results are stored in your working directory.

        Videos location: ./media/
        Results location: ./results/

        To run this pipeline:
        1. Activate environment:
           cd /home/ubuntu/voyager-sdk
           source venv/bin/activate

        2. Run inference with maximum performance:
           ./inference.py {selected_model} $PWD/media/{selected_video} \\
               --pipe gst \\
               --aipu-cores 4 \\
               --enable-hardware-codec

        3. Save results to local directory:
           ./inference.py {selected_model} $PWD/media/{selected_video} \\
               --output $PWD/results/output.mp4 \\
               --save-tracers $PWD/results/metrics.csv \\
               --pipe gst \\
               --aipu-cores 4 \\
               --enable-hardware-codec

        4. Use your own video (place in media folder):
           ./inference.py {selected_model} $PWD/media/your_video.mp4 \\
               --pipe gst --aipu-cores 4 --enable-hardware-codec

        5. Use USB camera with live display:
           ./inference.py {selected_model} usb:0 \\
               --pipe gst --aipu-cores 4

        Available videos in ./media/:
        - intersection_road.mp4 (general objects, traffic)
        - traffic1_720p.mp4 (traffic monitoring)
        - dancing.mp4 (people detection)
        - And more...

        Press 'Q' to quit the display window.
        """
    },
    tags=["pipeline", "deployment", detection_target, "local", "optimized"]
)

mcp__chipos__update_task(task_id=task3_id, status="done")
```

**⚡ Documentation is now complete! Deployment can run in background.**

**Step 2.4: Deploy Model (Task 4) - SMART SKIP ⚡**
```python
mcp__chipos__update_task(task_id=task4_id, status="doing")
```

```bash
# Activate environment
cd /home/ubuntu/voyager-sdk
source venv/bin/activate

# Check if already deployed (from Phase 0)
if [ "$SKIP_DEPLOYMENT" = "true" ]; then
    echo "✓ Model already deployed - skipping compilation!"
    echo "✓ Saved ~5 minutes!"
else
    echo "Deploying model: {selected_model} (~5 minutes)..."
    echo "Hardware mode: $([ "$METIS_AVAILABLE" = "true" ] && echo "Metis PCIe" || echo "CPU")"

    # Deploy with automatic hardware selection and maximum optimization
    if [ "$METIS_AVAILABLE" = "true" ]; then
        # Use all 4 AIPU cores, GStreamer pipeline, auto hardware acceleration
        ./deploy.py {selected_model} \
            --metis pcie \
            --aipu-cores 4 \
            --pipe gst \
            --mode PREQUANTIZED
    else
        # CPU mode with default settings
        ./deploy.py {selected_model}
    fi

    # Verify deployment succeeded
    if [ $? -eq 0 ]; then
        echo "✓ Deployment exit code: 0"

        # CRITICAL FIX #8: Verify .axnet file actually exists
        AXNET_FILE="/home/ubuntu/voyager-sdk/build/{selected_model}/*.axnet"
        if ls $AXNET_FILE 1> /dev/null 2>&1; then
            echo -n "✓ Model file verified: "
            ls -lh $AXNET_FILE | head -1 | awk '{print $9, $5}'
        else
            echo "ERROR: Deployment succeeded but .axnet file not found!"
            echo "Expected: $AXNET_FILE"
            echo "This usually means compilation failed silently."
            exit 1
        fi
    else
        echo "ERROR: Deployment failed - check logs"
        echo "TIP: Try running manually: cd /home/ubuntu/voyager-sdk && ./deploy.py {selected_model}"
        exit 1
    fi
fi
```

```python
mcp__chipos__update_task(task_id=task4_id, status="done")
```

**Step 2.5: Run Fast Demo Inference with Visualization (Task 5) 🎬**
```python
mcp__chipos__update_task(task_id=task5_id, status="doing")
```

**DEMO MODE: Use fastest video for quick results!**

Select video based on detection target:
- Animal detection: `media/traffic1_480p.mp4` (fastest, 16 MB, has vehicles+people)
- Traffic: `media/traffic1_480p.mp4` (8 MB, perfect for demo)
- Person: `media/dancing.mp4` (use 480p if available)
- General: `media/traffic1_480p.mp4` (fastest)

**Prioritize 480p videos for demo speed!**

```bash
# Capture current working directory before changing to SDK
WORK_DIR=$(pwd)
echo "Working directory: $WORK_DIR"

# Activate SDK environment
cd /home/ubuntu/voyager-sdk
source venv/bin/activate

echo "=========================================="
echo "🎬 DEMO MODE: Fast Inference Starting..."
echo "Model: {selected_model}"
echo "=========================================="

# NOTE: Tracking is NOT supported via CLI flag (--tracker flag doesn't exist)
# Tracking must be configured in the YAML pipeline configuration if needed

# Run inference with explicit absolute path (constructed from WORK_DIR)
echo "Running inference..."
./inference.py {selected_model} "$WORK_DIR/media/traffic1_720p.mp4" \
    --output "$WORK_DIR/results/traffic_output.mp4" \
    --save-tracers "$WORK_DIR/results/traffic_metrics.csv" \
    --pipe gst \
    --aipu-cores 4 \
    --enable-hardware-codec \
    2>&1

INFERENCE_EXIT=$?

# CRITICAL FIX #11: Check inference exit code
echo ""
echo "Checking inference result..."

if [ $INFERENCE_EXIT -eq 124 ]; then
    echo "⚠ Inference timed out after 30 seconds (demo mode)"
    echo "This is expected for demo - output may be partial but should work"
elif [ $INFERENCE_EXIT -ne 0 ]; then
    echo "ERROR: Inference failed with exit code $INFERENCE_EXIT"
    echo "Common causes:"
    echo "  - Video file not found or corrupted"
    echo "  - Model not properly deployed"
    echo "  - Insufficient memory"
    echo "  - AIPU hardware issue"
    echo ""
    echo "Try: Check that video exists and model is deployed"
    cd "$WORK_DIR"
    exit 1
else
    echo "✓ Inference completed successfully (exit code: 0)"
fi

# Return to working directory
cd "$WORK_DIR"

echo ""
echo "=========================================="
echo "✓ Inference Complete!"
echo "=========================================="
```

**Step 2.4.1: QUICK SANITY CHECK ✅**
```bash
# Verify output files were created (using relative paths)
echo ""
echo "Verifying output files..."

if [ -f "results/traffic_output.mp4" ]; then
    du -h "results/traffic_output.mp4" | cut -f1 | xargs echo "✓ Output video:"
else
    echo "ERROR: Output video not created!"
    exit 1
fi

if [ -f "results/traffic_metrics.csv" ]; then
    echo "✓ Metrics CSV created"
    # Show last line of metrics
    tail -1 "results/traffic_metrics.csv"
else
    echo "⚠ Metrics CSV not created (non-critical)"
fi

# Quick video verification with ffprobe
if command -v ffprobe &> /dev/null; then
    echo ""
    echo "Video properties:"
    ffprobe -v error -select_streams v:0 \
        -show_entries stream=width,height,nb_frames,duration \
        -of default=noprint_wrappers=1 "results/traffic_output.mp4" 2>&1 | head -5
fi

echo ""
echo "=========================================="
pwd | xargs echo "Results saved in:" | sed 's|$|/results/|'
ls -lh results/ | tail -5
echo "=========================================="
```

**Step 2.4.2: AUTO-PLAY RESULTS (Optional) 🎥**
```bash
# Try to open the video automatically (using relative path)
echo ""
echo "Opening results video..."

if command -v vlc &> /dev/null; then
    vlc "results/traffic_output.mp4" &
    echo "✓ Video opened in VLC"
elif command -v mpv &> /dev/null; then
    mpv "results/traffic_output.mp4" &
    echo "✓ Video opened in MPV"
elif command -v ffplay &> /dev/null; then
    ffplay -autoexit "results/traffic_output.mp4" &
    echo "✓ Video opened in ffplay"
else
    echo "⚠ No video player found. View manually:"
    pwd | xargs echo "   vlc" | sed 's|$|/results/traffic_output.mp4|'
fi

echo ""
echo "✓ Check the video for bounding boxes and labels!"
```

**What gets saved in results/:**
- `traffic_output.mp4` - Video with bounding boxes, labels, confidence scores
- `traffic_metrics.csv` - Performance metrics (FPS, latency, temp, CPU usage)
- Note: Fixed filenames (no timestamps to avoid bash substitution issues)

```python
mcp__chipos__update_task(task_id=task5_id, status="done")
```

**⚡ All tasks complete! Documentation was created in parallel with deployment.**

### Phase 2.6: Create Portable Package 📦🔌

**Package everything for USB/transfer portability**

```bash
echo ""
echo "=========================================="
echo "📦 Creating Portable Package"
echo "=========================================="

# Capture current directory
CURRENT_DIR=$(pwd)

# Create clean project name (no spaces)
DETECTION_TARGET="{detection_target}"
PROJECT_NAME_CLEAN=$(echo "$DETECTION_TARGET" | tr '[:upper:]' '[:lower:]' | tr ' ' '_' | tr -cd '[:alnum:]_')
PACKAGE_DIR="$CURRENT_DIR/results_${PROJECT_NAME_CLEAN}_detection"

echo "Package directory: $PACKAGE_DIR"

# Create directory structure
mkdir -p "$PACKAGE_DIR"/{pipeline,inputs/videos,outputs/{examples,latest},scripts,docs}

# Copy pipeline YAML from SDK
echo "Copying pipeline configuration..."
cp /home/ubuntu/voyager-sdk/ax_models/zoo/yolo/object_detection/{selected_model}.yaml \
   "$PACKAGE_DIR/pipeline/{selected_model}.yaml" 2>/dev/null || echo "Note: Pipeline YAML not copied (may not exist)"

# Create model info file
cat > "$PACKAGE_DIR/pipeline/model_info.txt" <<'MODELINFO'
model: {selected_model}
type: detection
classes: 80
input_size: 640x640
source: ultralytics
hardware: metis_pcie
aipu_cores: 4
pipeline: gst
MODELINFO

# Copy or reference videos
echo "Setting up input videos..."
if [ -f "media/traffic1_720p.mp4" ]; then
    cp "media/traffic1_720p.mp4" "$PACKAGE_DIR/inputs/videos/" 2>/dev/null || \
        echo "traffic1_720p.mp4 (from original location)" > "$PACKAGE_DIR/inputs/videos/videos.txt"
fi

# Copy example outputs
echo "Copying example outputs..."
cp "results/traffic_output.mp4" "$PACKAGE_DIR/outputs/examples/" 2>/dev/null || true
cp "results/traffic_metrics.csv" "$PACKAGE_DIR/outputs/examples/" 2>/dev/null || true
cp "results/traffic_log.txt" "$PACKAGE_DIR/outputs/examples/" 2>/dev/null || true

# Create project metadata
cat > "$PACKAGE_DIR/project_info.json" <<PROJINFO
{
  "project_name": "${PROJECT_NAME_CLEAN}_detection",
  "created_date": "$(date -Iseconds)",
  "created_by": "axe_command",
  "model": {
    "name": "{selected_model}",
    "type": "detection",
    "classes": 80,
    "input_size": [640, 640]
  },
  "hardware": {
    "target": "metis_pcie",
    "aipu_cores": 4,
    "pipeline": "gst"
  },
  "performance": {
    "fps": 660,
    "latency_ms": 33,
    "cpu_usage_percent": 35
  },
  "axe_command": "{$ARGUMENTS}"
}
PROJINFO

# Create scripts/setup.sh
cat > "$PACKAGE_DIR/scripts/setup.sh" <<'SETUPSCRIPT'
#!/bin/bash
# Setup script - verifies environment
set -e

PACKAGE_DIR="$(cd "$(dirname "$0")/.." && pwd)"

echo "Verifying environment..."

# Find Voyager SDK
if [ -z "$VOYAGER_SDK" ]; then
    if [ -d "/home/ubuntu/voyager-sdk" ]; then
        export VOYAGER_SDK="/home/ubuntu/voyager-sdk"
    elif [ -d "$HOME/voyager-sdk" ]; then
        export VOYAGER_SDK="$HOME/voyager-sdk"
    else
        echo "ERROR: Voyager SDK not found!"
        echo "Please set VOYAGER_SDK environment variable"
        exit 1
    fi
fi

cd "$VOYAGER_SDK"
source venv/bin/activate

echo "✓ Voyager SDK: $VOYAGER_SDK"
echo "✓ Python: $(which python)"
echo "✓ Package: $PACKAGE_DIR"

# Check hardware
if axdevice --pcie-scan 2>/dev/null | grep -q "metis"; then
    echo "✓ Metis hardware detected"
    export METIS_AVAILABLE=true
else
    echo "⚠ No Metis hardware - will use CPU"
    export METIS_AVAILABLE=false
fi

echo "✓ Setup complete!"
SETUPSCRIPT

# Create scripts/deploy.sh
cat > "$PACKAGE_DIR/scripts/deploy.sh" <<'DEPLOYSCRIPT'
#!/bin/bash
# Deploy model script
set -e

PACKAGE_DIR="$(cd "$(dirname "$0")/.." && pwd)"
MODEL=$(grep "^model:" "$PACKAGE_DIR/pipeline/model_info.txt" | cut -d: -f2 | tr -d ' ')

# Find SDK
source "$PACKAGE_DIR/scripts/setup.sh"

echo "Deploying model: $MODEL"

# Check if already deployed
if ls "$VOYAGER_SDK/build/$MODEL"/*.axnet 1>/dev/null 2>&1; then
    echo "✓ Model already deployed - skipping"
    exit 0
fi

# Deploy
cd "$VOYAGER_SDK"
echo "Deploying (may take 5 minutes)..."

if [ "$METIS_AVAILABLE" = "true" ]; then
    ./deploy.py "$MODEL" --metis pcie --aipu-cores 4 --pipe gst --mode PREQUANTIZED
else
    ./deploy.py "$MODEL"
fi

# Verify
if ls "$VOYAGER_SDK/build/$MODEL"/*.axnet 1>/dev/null 2>&1; then
    echo "✓ Model deployed successfully"
else
    echo "ERROR: Deployment failed"
    exit 1
fi
DEPLOYSCRIPT

# Create scripts/run_inference.sh
cat > "$PACKAGE_DIR/scripts/run_inference.sh" <<'INFSCRIPT'
#!/bin/bash
# Run inference script
set -e

PACKAGE_DIR="$(cd "$(dirname "$0")/.." && pwd)"
MODEL=$(grep "^model:" "$PACKAGE_DIR/pipeline/model_info.txt" | cut -d: -f2 | tr -d ' ')

# Find SDK
source "$PACKAGE_DIR/scripts/setup.sh"

# Find video
if [ -f "$PACKAGE_DIR/inputs/videos/traffic1_720p.mp4" ]; then
    VIDEO="$PACKAGE_DIR/inputs/videos/traffic1_720p.mp4"
else
    echo "ERROR: No video found in inputs/videos/"
    exit 1
fi

# Create output directory
mkdir -p "$PACKAGE_DIR/outputs/latest"

# Run inference
cd "$VOYAGER_SDK"
echo "Running inference with $MODEL..."

./inference.py "$MODEL" "$VIDEO" \
    --output "$PACKAGE_DIR/outputs/latest/output.mp4" \
    --save-tracers "$PACKAGE_DIR/outputs/latest/metrics.csv" \
    --pipe gst \
    --aipu-cores 4 \
    --enable-hardware-codec \
    2>&1 | tee "$PACKAGE_DIR/outputs/latest/log.txt"

echo ""
echo "✓ Inference complete!"
echo "Output: $PACKAGE_DIR/outputs/latest/output.mp4"
INFSCRIPT

# Create RUN_HERE.sh (one-click runner)
cat > "$PACKAGE_DIR/RUN_HERE.sh" <<'RUNSCRIPT'
#!/bin/bash
# One-click portable runner

set -e

echo "=========================================="
echo "🚀 Portable Pipeline Runner"
echo "=========================================="

PACKAGE_DIR="$(cd "$(dirname "$0")" && pwd)"

echo "Step 1: Setup environment..."
bash "$PACKAGE_DIR/scripts/setup.sh"

echo ""
echo "Step 2: Deploy model (if needed)..."
bash "$PACKAGE_DIR/scripts/deploy.sh"

echo ""
echo "Step 3: Run inference..."
bash "$PACKAGE_DIR/scripts/run_inference.sh"

echo ""
echo "=========================================="
echo "✓ Complete! Check outputs/latest/"
echo "=========================================="
RUNSCRIPT

# Create README.md
cat > "$PACKAGE_DIR/README.md" <<READMEEOF
# ${PROJECT_NAME_CLEAN} Detection Pipeline

**Quick Start (One Command):**
\`\`\`bash
./RUN_HERE.sh
\`\`\`

**Manual Steps:**
\`\`\`bash
./scripts/setup.sh      # Verify environment (one-time)
./scripts/deploy.sh     # Deploy model (~5 minutes, one-time)
./scripts/run_inference.sh  # Run inference
\`\`\`

## Package Contents

- **pipeline/**: Model configuration and YAML
- **inputs/**: Sample videos and inputs
- **outputs/**: Results (examples/ and latest/)
- **scripts/**: Portable run scripts
- **docs/**: Documentation

## Requirements

- Voyager SDK installed
- 500+ MB disk space
- Optional: Metis hardware for GPU acceleration (660 FPS vs CPU mode)

## Created With

\`\`\`
/axe {$ARGUMENTS}
\`\`\`

**Date:** $(date)
**Model:** {selected_model}
**Expected Performance:** 660 FPS (Metis), 33ms latency, 35% CPU

## Portability

This package is **fully portable**:
1. Copy entire directory to USB stick
2. Transfer to any computer with Voyager SDK
3. Run \`./RUN_HERE.sh\`
4. Works! ✅

**See docs/ for detailed instructions.**
READMEEOF

# Create docs/SETUP.md
cat > "$PACKAGE_DIR/docs/SETUP.md" <<'SETUPDOC'
# Setup Instructions

## Prerequisites

1. **Voyager SDK Installed**
   - Location: `/home/ubuntu/voyager-sdk` (or custom location)
   - Virtual environment activated
   - All dependencies installed

2. **System Requirements**
   - Linux (Ubuntu 20.04+ recommended)
   - 500+ MB disk space
   - Python 3.8+

3. **Optional: Metis Hardware**
   - For GPU acceleration (660 FPS)
   - Without Metis: CPU mode works but slower

## Installation Steps

### 1. Copy Package to Computer

\`\`\`bash
# From USB stick
cp -r /media/usb/results_animal_detection ~/

# Or download/transfer
cd ~/
tar -xzf results_animal_detection.tar.gz
\`\`\`

### 2. Set Voyager SDK Path (if needed)

\`\`\`bash
# If SDK is NOT at /home/ubuntu/voyager-sdk
export VOYAGER_SDK="/path/to/your/voyager-sdk"
\`\`\`

### 3. Run Setup Verification

\`\`\`bash
cd results_animal_detection
./scripts/setup.sh
\`\`\`

Should see:
\`\`\`
✓ Voyager SDK: /home/ubuntu/voyager-sdk
✓ Python: /home/ubuntu/voyager-sdk/venv/bin/python
✓ Metis hardware detected (or ⚠ No Metis - CPU mode)
✓ Setup complete!
\`\`\`

### 4. Deploy Model (One-Time)

\`\`\`bash
./scripts/deploy.sh
\`\`\`

Takes ~5 minutes first time. Subsequent runs skip this step.

### 5. Run Inference

\`\`\`bash
./scripts/run_inference.sh
\`\`\`

Results saved to: `outputs/latest/`

## Troubleshooting

**"Voyager SDK not found"**
- Set VOYAGER_SDK environment variable
- Or install SDK at /home/ubuntu/voyager-sdk

**"No Metis hardware"**
- This is OK! Pipeline runs in CPU mode
- Just slower than GPU mode

**"Model deployment failed"**
- Check disk space (need 500+ MB)
- Verify SDK installation is complete
- Try manual: `cd $VOYAGER_SDK && ./deploy.py model-name`

**"Video not found"**
- Check inputs/videos/ directory
- Add your own videos to inputs/videos/
- Or use USB camera: modify run_inference.sh to use `usb:0`
SETUPDOC

# Make scripts executable
chmod +x "$PACKAGE_DIR/RUN_HERE.sh"
chmod +x "$PACKAGE_DIR/scripts"/*.sh

echo ""
echo "✓ Portable package created successfully!"
echo ""
echo "Package: $PACKAGE_DIR"
echo -n "Size: "
du -sh "$PACKAGE_DIR" | cut -f1
echo ""
echo "To use on this machine:"
echo "  cd $PACKAGE_DIR"
echo "  ./RUN_HERE.sh"
echo ""
echo "To copy to USB stick:"
echo "  cp -r $PACKAGE_DIR /media/usb/"
echo ""
echo "On new computer with Voyager SDK:"
echo "  ./RUN_HERE.sh"
echo ""
echo "=========================================="
```

### Phase 3: Results Summary

**Present to user:**
```
✅ Pipeline Built Successfully!

Project: {project_title}
Model: {selected_model}
Detection Target: {detection_target}
Working Directory: (current directory)

Local Setup:
📁 Videos: ./media/ (16 sample videos ready)
📁 Results: ./results/ (outputs saved here)

Tasks Completed (Optimized Parallel Execution):
✓ Local environment setup (videos copied)
✓ Model verified
✓ Documentation created (in parallel with deployment! ⚡)
✓ Model deployed with maximum optimization
✓ Live inference complete with visualizations
✓ Portable package created 📦 (results_{detection_target}_detection/)

Quick Commands (Maximum Performance):
# Run pipeline with local video (use $PWD for absolute path):
cd /home/ubuntu/voyager-sdk
source venv/bin/activate
./inference.py {selected_model} $PWD/../yolov8-detect-and-track-voyager/media/{selected_video} \
    --pipe gst --aipu-cores 4 --enable-hardware-codec

# Save output to results folder:
./inference.py {selected_model} $PWD/../yolov8-detect-and-track-voyager/media/{selected_video} \
    --output $PWD/../yolov8-detect-and-track-voyager/results/output.mp4 \
    --save-tracers $PWD/../yolov8-detect-and-track-voyager/results/metrics.csv \
    --pipe gst --aipu-cores 4 --enable-hardware-codec

# Use your own video (place in media folder):
cp your_video.mp4 ./media/
./inference.py {selected_model} $PWD/../yolov8-detect-and-track-voyager/media/your_video.mp4 \
    --pipe gst --aipu-cores 4 --enable-hardware-codec

# Use USB camera:
./inference.py {selected_model} usb:0 --pipe gst --aipu-cores 4

Available Local Videos:
- intersection_road.mp4 (81 MB) - General objects, traffic
- traffic1_720p.mp4 (24 MB) - Traffic monitoring
- dancing.mp4 (67 MB) - People detection
- Plus 13 more videos in ./media/

📦 Portable Package Created:
- Location: ./results_{detection_target}_detection/
- One-click run: ./RUN_HERE.sh
- USB-ready: Copy entire directory to USB stick
- Works on any computer with Voyager SDK installed

Portable Package Usage:
# On this machine:
cd results_{detection_target}_detection
./RUN_HERE.sh

# Copy to USB and run on another machine:
cp -r results_{detection_target}_detection /media/usb/
# ... transfer to new machine ...
cd results_{detection_target}_detection
./RUN_HERE.sh

ChipOS Project ID: {project_id}
View tasks: mcp__chipos__list_tasks(filter_by="project", filter_value="{project_id}")
```

### Optimization Rules

1. **No User Questions** - Make smart decisions automatically
2. **Use Pre-Built Models** - Never create custom models
3. **Parallel Execution** - Run RAG queries in parallel (but expect empty results - see Known Issues)
4. **Maximum Hardware Performance**:
   - `--aipu-cores 4` - Use all 4 AIPU cores
   - `--pipe gst` - GStreamer pipeline (fastest C++ implementation)
   - `--enable-hardware-codec` - Hardware video encoding/decoding (VAAPI)
   - `--mode PREQUANTIZED` - Fastest deployment mode
5. **Local Storage** - Copy videos to current directory, save results locally
6. **Smart Video Selection** - Pick appropriate media from available samples
7. **NO Tracking via CLI** - Tracking flag `--tracker` doesn't exist; requires YAML config
8. **Task Updates** - Update ChipOS tasks in real-time
9. **Auto Hardware Detection** - Use `axdevice --pcie-scan` NOT `axdevice list`
10. **Zero Prompts** - Never ask for SDK path or permissions
11. **Careful Path Handling** - Use relative paths with explicit `cd` commands, or capture `$(pwd)` once and use quoted variables
12. **No Command Substitution in Pipelines** - Avoid `$(date)` and nested `$()` - they cause bash escaping errors
13. **Verify with axnet** - Deployed models are `.axnet` files, not `.axmodel`

### Detection Target → Model Mapping

| Detection Target | Best Pre-Built Model | COCO Classes |
|-----------------|---------------------|--------------|
| animal, dog, cat | yolov8n-coco | dog, cat, horse, cow, sheep, etc. |
| person, people, human | yolov8n-coco | person |
| traffic, car, vehicle | yolov8n-coco | car, truck, bus, motorcycle |
| object, general | yolov8n-coco | 80 classes |
| face | retinaface | face detection |
| pose, skeleton | yolov8npose-coco | 17 keypoints |

### Video Selection Logic

| Detection Target | Best Video | Reason |
|-----------------|-----------|--------|
| animal | intersection_road.mp4 | Has vehicles, people (general) |
| traffic | traffic1_720p.mp4 | Traffic scenes |
| person | dancing.mp4 | Human motion |
| general | intersection_road.mp4 | Mixed content |

### Example Prompts

```
/axe build animal detection with live visualization
/axe create person tracking system with live display
/axe traffic monitoring with vehicle detection
/axe object detection for USB camera
/axe face detection with live results
/axe build complete pipeline for animal detection using axelera hardware
```

## Important Notes

- **SDK Path**: Hardcoded to `/home/ubuntu/voyager-sdk` - NO permission prompts
- **Hardware**: Auto-detects Axelera Metis with `axdevice --pcie-scan` (NOT `axdevice list`)
- **No User Input**: Fully automated, zero questions asked
- **Fast Execution**: 45 seconds to 6 minutes depending on deployment
- **ChipOS Integration**: Full project and task management
- **RAG Research**: Parallel knowledge base queries (may return empty - proceed anyway)
- **Pre-Built Only**: No custom models or configs
- **Live Results**: Always shows display window and auto-plays results
- **Tracking**: NOT supported via `--tracker` CLI flag - must configure in YAML pipeline
- **Bash Limitations**: Avoid `$(date)` and complex variable substitutions - use absolute paths
- **Model Files**: Deployed models are `.axnet` files in `build/{model}/` directory
