# Build Native Components

Build GStreamer operators, trackers, and C/C++ examples for Axelera Metis AI hardware

## Important Context

**This command builds native components for the Voyager SDK framework.**
All components are built for deployment on Axelera Metis AI Processing Units (APUs).

## Instructions

Build the specified components: **$ARGUMENTS**

### Step 0: Data Source & Environment Selection

**FIRST**, use the `AskUserQuestion` tool to determine the setup:

```
Question: "Which Voyager SDK setup would you like to use for building components?"
Options:
1. "ChipOS Knowledge Base Only" - Use RAG for research and build guidance only (no local execution)
2. "ChipOS Knowledge Base + Local SDK" - Use RAG for build guidance, build in local SDK
3. "Local Voyager SDK Repository" - Use an existing local installation only
4. "Clone Fresh Repository" - Clone latest Voyager SDK for building
```

**Based on selection:**

**If ChipOS Knowledge Base Only:**
- Search for build information and documentation:
  ```python
  mcp__chipos__rag_search_knowledge_base(
      query="build operators trackers CMake make",
      source_id="github.com/axelera-ai-hub/voyager-sdk",
      match_count=5
  )
  mcp__chipos__rag_search_code_examples(
      query="CMakeLists build configuration",
      source_id="github.com/axelera-ai-hub/voyager-sdk",
      match_count=5
  )
  ```
- Provide research-based guidance and documentation
- No local SDK execution required - purely informational

**If ChipOS Knowledge Base + Local SDK:**
- Search for build instructions:
  ```python
  mcp__chipos__rag_search_knowledge_base(
      query="build operators trackers CMake make",
      source_id="github.com/axelera-ai-hub/voyager-sdk",
      match_count=5
  )
  ```
- Ask for local SDK path for building
- Set environment variables accordingly

**If Local Repository:**
- Ask for the repository path: "What is the path to your Voyager SDK installation?"
- Set environment:
  ```bash
  export AXELERA_FRAMEWORK=/path/to/voyager-sdk
  export AXELERA_RUNTIME_DIR=$AXELERA_FRAMEWORK/runtime
  source $AXELERA_FRAMEWORK/venv/bin/activate
  ```

**If Clone Fresh:**
- Clone and install:
  ```bash
  git clone https://github.com/axelera-ai-hub/voyager-sdk.git
  cd voyager-sdk
  ./install.sh --all --media
  source venv/bin/activate
  export AXELERA_FRAMEWORK=$(pwd)
  ```

### Step 0.5: ChipOS Project & Task Integration

**Check if project context is already set (from previous commands or `/init-chipos`).**

If NO project context exists, ask:
```
Question: "Would you like to track this build in ChipOS?"
Options:
1. "Yes, use existing project" - Select from available ChipOS projects
2. "Yes, create new project" - Create a new project for this work
3. "No, skip task tracking" - Proceed without ChipOS task management
```

**If using existing project:**
```python
mcp__chipos__list_projects()
# User selects project
```

**If creating new project:**
```python
mcp__chipos__create_project(
    title="Voyager SDK Build - [Component]",
    description="Build native components: $ARGUMENTS",
    github_repo="https://github.com/axelera-ai-hub/voyager-sdk"
)
```

**Create build task:**
```python
mcp__chipos__create_task(
    project_id="[PROJECT_ID]",
    title="Build: [Component from $ARGUMENTS]",
    description="Build native components: $ARGUMENTS",
    assignee="AI IDE Agent",
    feature="build",
    task_order=10
)
mcp__chipos__update_task(task_id="[TASK_ID]", status="doing")
```

### Step 1: Environment Verification
   - Verify Axelera environment is activated:
     ```bash
     if [ -z "$AXELERA_RUNTIME_DIR" ]; then
         echo "Please activate environment: source venv/bin/activate"
         exit 1
     fi
     ```

2. **Component Identification**
   Parse build target from arguments:
   - `all`: Build everything (operators, trackers, examples)
   - `operators`: Build GStreamer pipeline operators
   - `trackers`: Build object tracking algorithms
   - `examples`: Build C/C++ example applications
   - `clean`: Clean all build artifacts

3. **Build Operators**
   GStreamer operators provide pipeline processing capabilities:
   ```bash
   # Build all operators
   make operators

   # Or build directly from operators directory
   cd operators && make

   # Clear CMake cache if needed
   make clear-cmake-cache
   ```

   Operators built:
   - `axstreamer`: Core streaming operator
   - `gstaxstreamer`: GStreamer integration
   - `PillowResize`: Image resizing operator

4. **Build Trackers**
   Object tracking algorithms:
   ```bash
   # Build all trackers
   make trackers

   # Or build directly
   cd trackers && make
   ```

   Trackers built:
   - `ByteTrack`: Fast multi-object tracker
   - `OC-SORT`: Observation-centric SORT tracker

5. **Build Examples**
   C/C++ example applications:
   ```bash
   # Build all examples
   make examples

   # Requires operators to be built first
   make operators && make examples

   # Or build directly
   cd examples && make
   ```

   Example categories:
   - `axinferencenet/`: AxInferenceNet API examples
   - `axruntime/`: Low-level runtime API examples

6. **Build Everything**
   ```bash
   # Build all components
   make operators
   make trackers
   make examples
   ```

7. **CMake Build Options**
   For more control, use CMake directly:
   ```bash
   cd operators

   # Configure
   cmake -B build -DCMAKE_BUILD_TYPE=Release

   # Build
   cmake --build build -j$(nproc)

   # Install
   cmake --install build --prefix $AXELERA_FRAMEWORK/operators
   ```

8. **Debug Builds**
   ```bash
   # Build with debug symbols
   cd operators
   cmake -B build -DCMAKE_BUILD_TYPE=Debug
   cmake --build build
   ```

9. **Clean Build Artifacts**
   ```bash
   # Clean all libraries
   make clean-libs

   # Clobber (deep clean)
   make clobber-libs

   # Clean specific component
   cd operators && make clean
   cd trackers && make clean
   cd examples && make clean

   # Clear CMake cache
   make clear-cmake-cache
   ```

10. **Verify Build Success**
    Check that libraries are installed:
    ```bash
    # Check operator libraries
    ls -la $AXELERA_FRAMEWORK/operators/lib/

    # Check tracker libraries
    ls -la $AXELERA_FRAMEWORK/trackers/lib/

    # Check GStreamer plugins
    gst-inspect-1.0 axstreamer
    ```

11. **Troubleshooting Build Issues**

    **Missing runtime directory:**
    ```
    Error: Please set AXELERA_RUNTIME_DIR
    Solution: source venv/bin/activate
    ```

    **CMake configuration errors:**
    ```bash
    # Clear cache and reconfigure
    make clear-cmake-cache
    make operators
    ```

    **Missing dependencies:**
    ```bash
    # Install build dependencies
    sudo apt install build-essential cmake
    sudo apt install libgstreamer1.0-dev libgstreamer-plugins-base1.0-dev
    ```

    **Linker errors:**
    ```bash
    # Check library paths
    echo $LD_LIBRARY_PATH
    # Add runtime library path
    export LD_LIBRARY_PATH=$AXELERA_RUNTIME_DIR/lib:$LD_LIBRARY_PATH
    ```

12. **Build Configuration**
    The build uses these configuration files:
    - `operators/CMakeLists.txt`: Operator build config
    - `trackers/CMakeLists.txt`: Tracker build config
    - `examples/CMakeLists.txt`: Examples build config
    - `Makefile`: Top-level build orchestration

13. **Model-Specific Builds**
    For model-specific components:
    ```bash
    # Show available models
    make help

    # Build for specific model (if applicable)
    make NN=yolov5s-v7-coco <target>

    # Show model info
    make NN=yolov5s-v7-coco info
    ```

14. **Docker Build (Alternative)**
    If build fails locally, use Docker:
    ```bash
    ./launch-docker.sh
    # Inside container:
    make operators
    make trackers
    ```
