# Deploy Model Pipeline

Compile and deploy a model pipeline for Axelera Metis AI hardware

## Important Context

**This command deploys AI models to Axelera Metis AI Processing Units (APUs).**
All deployments target Metis hardware using the Voyager SDK compilation toolchain.

## Instructions

Deploy the specified model/pipeline: **$ARGUMENTS**

### Step 0: Data Source & Environment Selection

**FIRST**, use the `AskUserQuestion` tool to determine the setup:

```
Question: "Which Voyager SDK setup would you like to use?"
Options:
1. "ChipOS Knowledge Base Only" - Use RAG for research and deployment guidance only (no local execution)
2. "ChipOS Knowledge Base + Local SDK" - Use RAG for guidance, execute on local SDK
3. "Local Voyager SDK Repository" - Use an existing local installation only
4. "Clone Fresh Repository" - Clone and set up latest Voyager SDK from GitHub
```

**Based on selection:**

**If ChipOS Knowledge Base Only:**
- Search for deployment information and documentation:
  ```python
  mcp__chipos__rag_search_knowledge_base(
      query="deploy model compilation",
      source_id="github.com/axelera-ai-hub/voyager-sdk",
      match_count=5
  )
  mcp__chipos__rag_search_code_examples(
      query="deployment configuration YAML",
      source_id="github.com/axelera-ai-hub/voyager-sdk",
      match_count=5
  )
  ```
- Provide research-based guidance and documentation
- No local SDK execution required - purely informational

**If ChipOS Knowledge Base + Local SDK:**
- Use `mcp__chipos__rag_search_knowledge_base(query="deploy model", source_id="github.com/axelera-ai-hub/voyager-sdk")` for guidance
- Ask for local SDK path for execution
- Set environment variables accordingly

**If Local Repository:**
- Ask for the repository path: "What is the path to your Voyager SDK installation?"
- Set environment:
  ```bash
  export AXELERA_FRAMEWORK=/path/to/voyager-sdk
  export AXELERA_RUNTIME_DIR=$AXELERA_FRAMEWORK/runtime
  source $AXELERA_FRAMEWORK/venv/bin/activate
  ```

**If Clone Fresh:**
- Clone and install:
  ```bash
  git clone https://github.com/axelera-ai-hub/voyager-sdk.git
  cd voyager-sdk
  ./install.sh --all --media
  source venv/bin/activate
  export AXELERA_FRAMEWORK=$(pwd)
  ```

### Step 0.5: ChipOS Project & Task Integration

**Check if project context is already set (from previous commands or `/init-chipos`).**

If NO project context exists, ask:
```
Question: "Would you like to track this deployment in ChipOS?"
Options:
1. "Yes, use existing project" - Select from available ChipOS projects
2. "Yes, create new project" - Create a new project for this work
3. "No, skip task tracking" - Proceed without ChipOS task management
```

**If using existing project:**
```python
mcp__chipos__list_projects()
# User selects project
```

**If creating new project:**
```python
mcp__chipos__create_project(
    title="Voyager SDK Deployment - [Model Name]",
    description="Model deployment: $ARGUMENTS",
    github_repo="https://github.com/axelera-ai-hub/voyager-sdk"
)
```

**Create deployment task:**
```python
mcp__chipos__create_task(
    project_id="[PROJECT_ID]",
    title="Deploy: [Model name from $ARGUMENTS]",
    description="Compile and deploy model for Metis hardware: $ARGUMENTS",
    assignee="AI IDE Agent",
    feature="deployment",
    task_order=10
)
mcp__chipos__update_task(task_id="[TASK_ID]", status="doing")
```

### Step 1: Pre-Deployment Checks
   - Verify Axelera environment is activated:
     ```bash
     echo $AXELERA_FRAMEWORK
     ```
   - If not set, activate with: `source venv/bin/activate`
   - Verify Metis hardware is connected:
     ```bash
     axdevice list
     ```

2. **Model/Pipeline Identification**
   - Parse model name from arguments
   - If no model specified, list available models:
     ```bash
     ./deploy.py --help
     ```
   - Locate the YAML configuration in `ax_models/`
   - Verify YAML file exists and is valid

3. **Deployment Mode Selection**
   Choose appropriate deployment mode:
   - `--mode compile` (default): Full compilation for Metis
   - `--mode quantize`: Only quantize the model (for debugging)
   - `--mode quantize_debug`: Quantize with debug outputs
   - `--mode export`: Export to different format

4. **Target Configuration**
   - `--metis m2`: Target Metis M.2 device (default)
   - `--metis pcie`: Target Metis PCIe device
   - `--aipu-cores <n>`: Number of AIPU cores to use (1-4)

5. **Deployment Execution**
   Run deployment command:
   ```bash
   ./deploy.py <model-name> [options]
   ```

   Common examples:
   ```bash
   # Basic deployment
   ./deploy.py yolov5s-v7-coco

   # Deploy with specific model variant
   ./deploy.py yolov8n-coco --model yolov8n

   # Quantize only (for debugging)
   ./deploy.py yolov5s-v7-coco --mode quantize

   # Deploy for PCIe with 4 cores
   ./deploy.py yolov8m-coco --metis pcie --aipu-cores 4

   # Deploy specific components
   ./deploy.py yolov5s-v7-coco --pipeline-only
   ./deploy.py yolov5s-v7-coco --models-only
   ```

6. **Calibration Process**
   - Deployment includes automatic quantization calibration
   - Calibration uses dataset specified in YAML
   - For reproducible results, set calibration seed:
     ```bash
     ./deploy.py <model> --cal-seed 42
     ```

7. **Build Output Verification**
   After successful deployment, verify outputs:
   - Build directory: `build/<model-name>/<arch>/`
   - Compiled model: `*.axmodel` files
   - Pipeline configuration: `pipeline.yaml`
   - Check compilation logs for warnings

8. **Troubleshooting Common Issues**

   **Environment not activated:**
   ```
   Error: Please set AXELERA_FRAMEWORK variable
   Solution: source venv/bin/activate
   ```

   **Model not found:**
   ```
   Error: NN=<name> does not specify a valid model
   Solution: Check model name with ./deploy.py --help
   ```

   **Calibration data missing:**
   ```
   Error: Calibration dataset not found
   Solution: Download dataset or specify --cal-data path
   ```

   **Out of memory during compilation:**
   ```
   Solution: Reduce batch size or model complexity
   ```

9. **Post-Deployment Validation**
   - Test the deployed model:
     ```bash
     ./inference.py <model-name> <test-input> --no-display
     ```
   - Check model accuracy:
     ```bash
     ./inference.py <model-name> dataset --no-display
     ```

10. **Advanced Options**
    ```bash
    # Verbose logging
    ./deploy.py <model> --log-level debug

    # Custom output directory
    ./deploy.py <model> --output-dir /path/to/output

    # Export ONNX
    ./deploy.py <model> --export onnx
    ```
