# Create New Application

Create a Python application for Axelera Metis AI hardware using Voyager SDK InferenceStream

## Important Context

**This command creates applications specifically for Axelera Metis AI Processing Units (APUs).**
All applications use the Voyager SDK InferenceStream API to run inference on Metis hardware.

## Instructions

Create an application with the specified requirements: **$ARGUMENTS**

### Step 0: Data Source & Environment Selection

**FIRST**, use the `AskUserQuestion` tool to determine the setup:

```
Question: "Which Voyager SDK setup would you like to use for application development?"
Options:
1. "ChipOS Knowledge Base Only" - Use RAG for research and API examples only (no local execution)
2. "ChipOS Knowledge Base + Local SDK" - Use RAG for API examples, develop in local SDK
3. "Local Voyager SDK Repository" - Use an existing local installation only
4. "Clone Fresh Repository" - Clone latest Voyager SDK for new app development
```

**Based on selection:**

**If ChipOS Knowledge Base Only:**
- Search for application development information and documentation:
  ```python
  mcp__chipos__rag_search_code_examples(
      query="InferenceStream application",
      source_id="github.com/axelera-ai-hub/voyager-sdk",
      match_count=5
  )
  mcp__chipos__rag_search_knowledge_base(
      query="create_inference_stream API patterns",
      source_id="github.com/axelera-ai-hub/voyager-sdk",
      match_count=5
  )
  ```
- Provide research-based guidance and documentation
- No local SDK execution required - purely informational

**If ChipOS Knowledge Base + Local SDK:**
- Search for InferenceStream API examples:
  ```python
  mcp__chipos__rag_search_code_examples(
      query="InferenceStream application",
      source_id="github.com/axelera-ai-hub/voyager-sdk",
      match_count=5
  )
  mcp__chipos__rag_search_knowledge_base(
      query="create_inference_stream API",
      source_id="github.com/axelera-ai-hub/voyager-sdk",
      match_count=5
  )
  ```
- Ask for local SDK path to create the application
- Set environment variables accordingly

**If Local Repository:**
- Ask for the repository path: "What is the path to your Voyager SDK installation?"
- Set environment:
  ```bash
  export AXELERA_FRAMEWORK=/path/to/voyager-sdk
  export AXELERA_RUNTIME_DIR=$AXELERA_FRAMEWORK/runtime
  source $AXELERA_FRAMEWORK/venv/bin/activate
  ```
- Review existing examples in `$AXELERA_FRAMEWORK/examples/`

**If Clone Fresh:**
- Clone and install:
  ```bash
  git clone https://github.com/axelera-ai-hub/voyager-sdk.git
  cd voyager-sdk
  ./install.sh --all --media
  source venv/bin/activate
  export AXELERA_FRAMEWORK=$(pwd)
  ```

### Step 0.5: ChipOS Project & Task Integration

**Check if project context is already set (from previous commands or `/init-chipos`).**

If NO project context exists, ask:
```
Question: "Would you like to track this application development in ChipOS?"
Options:
1. "Yes, use existing project" - Select from available ChipOS projects
2. "Yes, create new project" - Create a new project for this work
3. "No, skip task tracking" - Proceed without ChipOS task management
```

**If using existing project:**
```python
mcp__chipos__list_projects()
# User selects project
```

**If creating new project:**
```python
mcp__chipos__create_project(
    title="Voyager SDK App - [App Description]",
    description="Application development: $ARGUMENTS",
    github_repo="https://github.com/axelera-ai-hub/voyager-sdk"
)
```

**Create application development task:**
```python
mcp__chipos__create_task(
    project_id="[PROJECT_ID]",
    title="Create app: [App type from $ARGUMENTS]",
    description="Develop application using InferenceStream API: $ARGUMENTS",
    assignee="AI IDE Agent",
    feature="application-development",
    task_order=10
)
mcp__chipos__update_task(task_id="[TASK_ID]", status="doing")
```

### Step 1: Application Requirements Analysis
   - Parse application type from arguments
   - If not specified, ask for:
     - Application purpose (monitoring, analytics, demo, production)
     - Model/pipeline to use
     - Input sources (cameras, video files, streams)
     - Output requirements (display, file, API, custom processing)
     - Business logic requirements

2. **Application Template Selection**
   Based on requirements, choose appropriate template:
   - **Simple Demo**: Basic display with inference results
   - **Analytics**: Data collection and metrics
   - **Monitoring**: Multi-stream with alerts
   - **Custom Processing**: Full control over inference results
   - **REST API**: HTTP endpoint for inference

3. **Basic Application Structure**
   Create application file with this structure:
   ```python
   #!/usr/bin/env python
   # Copyright <year>
   # Application: <description>

   from axelera.app import config, create_inference_stream, display

   # Configuration
   MODEL = "<model-name>"
   SOURCES = ["<source1>", "<source2>"]

   # Create inference stream
   stream = create_inference_stream(
       network=MODEL,
       sources=SOURCES,
   )

   def process_frame(frame_result):
       """Process a single inference result."""
       image = frame_result.image
       meta = frame_result.meta
       stream_id = frame_result.stream_id

       # Add custom processing here
       pass

   def main(window, stream):
       """Main inference loop."""
       for frame_result in stream:
           process_frame(frame_result)
           window.show(frame_result.image, frame_result.meta, frame_result.stream_id)

           if window.is_closed:
               break

   # Run application
   with display.App(renderer=True) as app:
       wnd = app.create_window("Application Title", (1280, 720))
       app.start_thread(main, (wnd, stream), name='InferenceThread')
       app.run()

   stream.stop()
   ```

4. **Working with Detection Results**
   ```python
   # Access detection metadata
   for frame_result in stream:
       # Iterate over detected objects
       for detection in frame_result.detections:
           bbox = detection.bbox  # [x1, y1, x2, y2]
           label = detection.label.name
           confidence = detection.confidence
           track_id = detection.track_id  # If tracker enabled

           print(f"{label}: {confidence:.2f} at {bbox}")

       # Access specific detection types
       for person in frame_result.persons:
           print(f"Person: {person.bbox}")

       for vehicle in frame_result.vehicles:
           print(f"Vehicle: {vehicle.label.name}")
   ```

5. **Working with Pose Estimation**
   ```python
   for frame_result in stream:
       for pose in frame_result.poses:
           keypoints = pose.keypoints  # List of (x, y, confidence)
           bbox = pose.bbox

           # Access specific keypoints
           nose = pose.nose
           left_shoulder = pose.left_shoulder
           right_shoulder = pose.right_shoulder
   ```

6. **Working with Segmentation**
   ```python
   for frame_result in stream:
       segmentation_mask = frame_result.segmentation_mask
       # mask shape: (H, W) with class IDs

       # Get colored visualization
       colored_mask = frame_result.segmentation_colored
   ```

7. **Working with Classification**
   ```python
   for frame_result in stream:
       predictions = frame_result.classifications
       for pred in predictions:
           label = pred.label
           confidence = pred.confidence
           print(f"{label}: {confidence:.2%}")
   ```

8. **Multi-Stream Handling**
   ```python
   stream = create_inference_stream(
       network="yolov8n-coco",
       sources=[
           "usb:0",
           "media/video1.mp4",
           "rtsp://camera/stream"
       ],
   )

   def main(window, stream):
       # Configure per-stream display
       window.options(0, title="Camera 1")
       window.options(1, title="Video Feed")
       window.options(2, title="RTSP Stream")

       for frame_result in stream:
           stream_id = frame_result.stream_id

           # Process based on stream
           if stream_id == 0:
               # Camera-specific logic
               pass
           elif stream_id == 1:
               # Video-specific logic
               pass

           window.show(frame_result.image, frame_result.meta, stream_id)
   ```

9. **Headless Processing (No Display)**
   ```python
   from axelera.app import create_inference_stream

   stream = create_inference_stream(
       network="yolov8n-coco",
       sources=["video.mp4"],
   )

   results = []
   for frame_result in stream:
       # Process without display
       frame_data = {
           "frame_id": frame_result.frame_id,
           "detections": [
               {
                   "label": d.label.name,
                   "bbox": d.bbox.tolist(),
                   "confidence": float(d.confidence)
               }
               for d in frame_result.detections
           ]
       }
       results.append(frame_data)

   stream.stop()

   # Save results
   import json
   with open("results.json", "w") as f:
       json.dump(results, f, indent=2)
   ```

10. **REST API Application**
    ```python
    from flask import Flask, jsonify, request
    from axelera.app import create_inference_stream
    import threading

    app = Flask(__name__)

    # Global inference stream
    stream = None
    latest_results = {}

    def inference_worker():
        global latest_results
        for frame_result in stream:
            latest_results = {
                "frame_id": frame_result.frame_id,
                "detections": [
                    {"label": d.label.name, "confidence": float(d.confidence)}
                    for d in frame_result.detections
                ]
            }

    @app.route("/start", methods=["POST"])
    def start_inference():
        global stream
        source = request.json.get("source", "usb:0")
        stream = create_inference_stream(network="yolov8n-coco", sources=[source])
        threading.Thread(target=inference_worker, daemon=True).start()
        return jsonify({"status": "started"})

    @app.route("/results", methods=["GET"])
    def get_results():
        return jsonify(latest_results)

    @app.route("/stop", methods=["POST"])
    def stop_inference():
        if stream:
            stream.stop()
        return jsonify({"status": "stopped"})

    if __name__ == "__main__":
        app.run(host="0.0.0.0", port=5000)
    ```

11. **Event-Based Processing**
    ```python
    from axelera.app import create_inference_stream, FrameEventType

    stream = create_inference_stream(network="yolov8n-coco", sources=["video.mp4"])

    for event in stream.with_events():
        if event.type == FrameEventType.FRAME:
            # Normal frame result
            process_frame(event.result)
        elif event.type == FrameEventType.EOS:
            # End of stream
            print("Stream ended")
            break
        elif event.type == FrameEventType.ERROR:
            # Error occurred
            print(f"Error: {event.error}")
    ```

12. **Output File Location**
    Save application to:
    - Examples: `examples/<app-name>.py`
    - Production: Custom location specified by user
    - Ensure executable permissions: `chmod +x <app-name>.py`

13. **Testing the Application**
    ```bash
    # Run the application
    python examples/<app-name>.py

    # Or make executable and run
    chmod +x examples/<app-name>.py
    ./examples/<app-name>.py
    ```
