# Create New AI Pipeline

Create a complete YAML pipeline configuration for Axelera Metis hardware using Voyager SDK

## Important Context

**This command creates AI pipelines specifically for Axelera Metis AI Processing Units (APUs).**
All pipelines are designed for deployment on Metis hardware using the Voyager SDK framework.

## Instructions

Follow these steps to create a new AI pipeline: **$ARGUMENTS**

### Step 0: Data Source & Environment Selection

**FIRST**, use the `AskUserQuestion` tool to determine the setup:

```
Question: "Which Voyager SDK setup would you like to use?"
Options:
1. "ChipOS Knowledge Base Only" - Use RAG for research and pipeline examples only (no local execution)
2. "ChipOS Knowledge Base + Local SDK" - Use RAG for pipeline examples, create in local SDK
3. "Local Voyager SDK Repository" - Use an existing local installation only
4. "Clone Fresh Repository" - Clone latest Voyager SDK for new pipeline development
```

**Based on selection:**

**If ChipOS Knowledge Base Only:**
- Search for pipeline information and documentation:
  ```python
  mcp__chipos__rag_search_code_examples(
      query="pipeline YAML configuration",
      source_id="github.com/axelera-ai-hub/voyager-sdk",
      match_count=5
  )
  mcp__chipos__rag_search_knowledge_base(
      query="pipeline preprocessing postprocessing operators",
      source_id="github.com/axelera-ai-hub/voyager-sdk",
      match_count=5
  )
  ```
- Provide research-based guidance and documentation
- No local SDK execution required - purely informational

**If ChipOS Knowledge Base + Local SDK:**
- Search for pipeline examples:
  ```python
  mcp__chipos__rag_search_code_examples(
      query="pipeline YAML configuration",
      source_id="github.com/axelera-ai-hub/voyager-sdk",
      match_count=5
  )
  mcp__chipos__rag_search_knowledge_base(
      query="pipeline preprocessing postprocessing",
      source_id="github.com/axelera-ai-hub/voyager-sdk",
      match_count=5
  )
  ```
- Ask for local SDK path to create the pipeline file
- Set environment variables accordingly

**If Local Repository:**
- Ask for the repository path: "What is the path to your Voyager SDK installation?"
- Set environment:
  ```bash
  export AXELERA_FRAMEWORK=/path/to/voyager-sdk
  export AXELERA_RUNTIME_DIR=$AXELERA_FRAMEWORK/runtime
  source $AXELERA_FRAMEWORK/venv/bin/activate
  ```
- Review existing pipelines in `$AXELERA_FRAMEWORK/ax_models/zoo/`

**If Clone Fresh:**
- Clone and install:
  ```bash
  git clone https://github.com/axelera-ai-hub/voyager-sdk.git
  cd voyager-sdk
  ./install.sh --all --media
  source venv/bin/activate
  export AXELERA_FRAMEWORK=$(pwd)
  ```

### Step 0.5: ChipOS Project & Task Integration

**Check if project context is already set (from previous commands or `/init-chipos`).**

If NO project context exists, ask:
```
Question: "Would you like to track this pipeline creation in ChipOS?"
Options:
1. "Yes, use existing project" - Select from available ChipOS projects
2. "Yes, create new project" - Create a new project for this work
3. "No, skip task tracking" - Proceed without ChipOS task management
```

**If using existing project:**
```python
mcp__chipos__list_projects()
# User selects project
```

**If creating new project:**
```python
mcp__chipos__create_project(
    title="Voyager SDK Pipeline - [Pipeline Type]",
    description="Create new pipeline: $ARGUMENTS",
    github_repo="https://github.com/axelera-ai-hub/voyager-sdk"
)
```

**Create pipeline task:**
```python
mcp__chipos__create_task(
    project_id="[PROJECT_ID]",
    title="Create pipeline: [Pipeline type from $ARGUMENTS]",
    description="Design and implement YAML pipeline: $ARGUMENTS",
    assignee="AI IDE Agent",
    feature="pipeline-development",
    task_order=10
)
mcp__chipos__update_task(task_id="[TASK_ID]", status="doing")
```

### Step 1: Pipeline Requirements Analysis
   - Parse the pipeline type from arguments (detection, classification, segmentation, pose, cascade, parallel)
   - If no arguments provided, ask user for:
     - Pipeline purpose (object detection, classification, segmentation, pose estimation, etc.)
     - Target model from model zoo or custom model
     - Input source type (camera, video file, image, RTSP stream)
     - Output requirements (display, file, metadata only)

2. **Model Selection**
   - Review available models in `ax_models/zoo/` directory
   - For detection: YOLOv5, YOLOv8, YOLOv10, SSD-MobileNet, RetinaFace
   - For classification: ResNet, EfficientNet, MobileNet, DenseNet, Inception
   - For segmentation: YOLOv8-seg, UNet
   - For pose estimation: YOLOv8-pose
   - For depth: FastDepth
   - For LLM: Llama, Phi3
   - Match model to user requirements (accuracy vs speed trade-off)

3. **Pipeline YAML Structure**
   Create YAML file in `ax_models/` with this structure:
   ```yaml
   # Pipeline: <descriptive-name>
   # Description: <what this pipeline does>

   network:
     name: <model-name>

   model:
     source: <torch|onnx|timm|ultralytics>
     # Model-specific configuration

   input:
     shape: [1, 3, <height>, <width>]
     layout: NCHW
     dtype: float32

   preprocessing:
     # Define preprocessing operators

   postprocessing:
     # Define postprocessing/decoder operators

   calibration:
     # Calibration data configuration
   ```

4. **Preprocessing Configuration**
   Based on model requirements, configure:
   - Image resizing (letterbox, stretch, crop)
   - Color space conversion (RGB, BGR)
   - Normalization (mean, std values)
   - Data type conversion

   Example operators:
   ```yaml
   preprocessing:
     - type: AxResize
       mode: letterbox
       size: [640, 640]
     - type: AxNormalize
       mean: [0.485, 0.456, 0.406]
       std: [0.229, 0.224, 0.225]
   ```

5. **Postprocessing/Decoder Configuration**
   Configure appropriate decoder based on model type:
   - Detection: YOLO decoder, SSD decoder, RetinaFace decoder
   - Classification: Softmax, Top-K
   - Segmentation: Argmax, threshold
   - Pose: Keypoint decoder

   Example:
   ```yaml
   postprocessing:
     - type: YoloDecoder
       conf_threshold: 0.25
       iou_threshold: 0.45
       num_classes: 80
   ```

6. **Cascade Pipeline (if needed)**
   For multi-model pipelines:
   ```yaml
   pipeline:
     - stage: detection
       model: yolov8n
       output: bboxes

     - stage: classification
       model: resnet50
       input: cropped_bboxes
       trigger: detection_results
   ```

7. **Parallel Pipeline (if needed)**
   For running multiple models simultaneously:
   ```yaml
   parallel_pipelines:
     - name: pose_detection
       model: yolov8s-pose
     - name: face_detection
       model: retinaface
   ```

8. **Tracker Integration (if needed)**
   Add object tracking for video streams:
   ```yaml
   tracker:
     type: ByteTrack  # or OC-SORT
     track_thresh: 0.5
     track_buffer: 30
     match_thresh: 0.8
   ```

9. **Calibration Data Setup**
   Configure calibration for quantization:
   ```yaml
   calibration:
     dataset: coco2017
     num_samples: 100
     # Or custom calibration data:
     data_path: /path/to/calibration/images
   ```

10. **Validation and Testing**
    - Validate YAML syntax
    - Check model availability
    - Verify preprocessing matches model expectations
    - Test pipeline deployment:
      ```bash
      ./deploy.py <pipeline-name> --mode quantize
      ./inference.py <pipeline-name> <test-source>
      ```

11. **Output File Location**
    Save the pipeline YAML to appropriate location:
    - Simple models: `ax_models/zoo/<category>/<model-name>.yaml`
    - Reference pipelines: `ax_models/reference/<type>/<pipeline-name>.yaml`
    - Tutorial/custom: `ax_models/tutorials/<project>/<name>.yaml`

12. **Documentation**
    Add inline comments explaining:
    - Pipeline purpose and use case
    - Model source and expected accuracy
    - Input/output specifications
    - Any special requirements or limitations
